//---------------------------------------------------------
//--- DRAW FUNCTIONS --------------------------------------
//---------------------------------------------------------
/*
	To use theses drawing functions, there must be an "display" named "screen" :

	<display name="screen" width="320" height="240" embeed="true" monitorpos="0" monitorscale="1.5"/> 

*/

#include "math.h"

void clear() {
    screen.setBackground( 0 );
}


void drawLine(double x0, double y0, double x1, double y1, uint color) {
	//=== Tracer une ligne ===
	
	int x = int(x0);
	int y = int(y0);
	int xend = int(x1);
	int yend = int(y1);

    int dx = abs(xend - x);
    int dy = abs(yend - y);

    int sx = x0 < x1 ? 1 : -1;
    int sy = y0 < y1 ? 1 : -1;

    int err = dx - dy;
	
    while (true) {
        screen.setPixel(x, y, color);

        if (x == xend && y == yend)
            break;

        int e2 = 2 * err;

        if (e2 > -dy) {
            err -= dy;
            x += sx;
        }

        if (e2 < dx) {
            err += dx;
            y += sy;
        }
    }
}


void drawPolygon(array<array<double>> &in polygone, uint color, bool close) {
	//=== Tracer un polygone ===
	
    uint n = polygone.length();
    if (n < 2) return; // Besoin d'au moins deux points pour tracer des lignes

    for (uint i = 1; i < n; i++) {
		double x0 = polygone[i-1][0];
		double y0 = polygone[i-1][1];
        double x1 = polygone[i][0]; // Boucle vers le premier sommet
        double y1 = polygone[i][1];

        drawLine(int(x0), int(y0), int(x1), int(y1), color);
    }
	if (close) {
		double x0 = polygone[n-1][0];
		double y0 = polygone[n-1][1];
        double x1 = polygone[0][0]; // Boucle vers le premier sommet
        double y1 = polygone[0][1];
		drawLine(int(x0), int(y0), int(x1), int(y1), color);
	}
}


void drawArc(double xc, double yc, double radius, double startAngle, double endAngle, uint color) {
	//=== Tracer un arc de cercle ===
    
    array<array<double>> arcPoints;
	const double PI2 = PI*2.0;

	startAngle = fmod(startAngle, PI2);
	endAngle = fmod(endAngle, PI2);
	
	while (endAngle < startAngle) endAngle += PI2;
	double len = (endAngle-startAngle)*radius;
	int n = ceil(len / 5.0);
	double da = (endAngle-startAngle)/n;
	
	for (double a=startAngle; a <= endAngle+da/5; a+=da) {
		double x = xc + radius * cos(a);
		double y = yc - radius * sin(a);
		arcPoints.insertLast({x, y});
		//print(""+a+":"+x+","+y);
	}

    // Tracer l'arc en tant que polygone
    drawPolygon(arcPoints, color, false);
}


void fillPolygon(array<array<double>> &in polygone, uint color)
{
    // S'assurer qu'il y a au moins 3 points
    if (polygone.size() < 3)
        return;

    // Extraire les sommets (x, y)
    // On suppose polygone[i][0] = x, polygone[i][1] = y
    int n = int(polygone.size());

    // Trouver le minY et le maxY
    double minY = polygone[0][1];
    double maxY = polygone[0][1];
    for (int i = 1; i < n; i++)
    {
        double py = polygone[i][1];
        if (py < minY) minY = py;
        if (py > maxY) maxY = py;
    }

    // Convertir en entiers pour le parcours des scanlines
    int startY = int(floor(minY));
    int endY   = int(ceil(maxY));

    // Parcours de chaque ligne horizontale
    for (int y = startY; y <= endY; y++)
    {
        // Liste des intersections pour cette ligne y
        array<double> intersections;

        for (int i = 0; i < n; i++)
        {
            int j = (i + 1) % n; // Point suivant
            double x1 = polygone[i][0];
            double y1 = polygone[i][1];
            double x2 = polygone[j][0];
            double y2 = polygone[j][1];

            // Vérifier si la ligne horizontale y intersecte le segment (x1,y1)-(x2,y2)
            // Conditions d'intersection :
            // Le segment doit franchir la ligne y : (y est entre y1 et y2)
            // Et ne pas être horizontal.
            if ((y1 <= y && y2 > y) || (y2 <= y && y1 > y))
            {
                // Calculer le point d'intersection sur x
                double dy = y2 - y1;
                if (abs(dy) < 1e-12) 
                    continue; // Evite division par zéro dans le cas très particulier d'une arrête horizontale.

                double t = (y - y1) / dy;
                double x_intersect = x1 + t * (x2 - x1);

                intersections.insertLast(x_intersect);
            }
        }

        // Trier les intersections par leur coordonnée x
        intersections.sortAsc();

        // Maintenant, remplir les pixels entre chaque paire d'intersections
        // On suppose un polygone fermé, donc le nombre d'intersections devrait être pair
        for (uint k = 0; k + 1 < intersections.size(); k += 2)
        {
            int startX = int(ceil(intersections[k]));
            int endX   = int(floor(intersections[k+1]));
            for (int x = startX; x <= endX; x++)
            {
                screen.setPixel(x, y, color);
            }
        }
    }
}


// Fonction interne pour dessiner une ligne horizontale
void drawHorizontalLine(int yy, int x1, int x2, uint col) {
	for (int X = x1; X <= x2; X++) {
		screen.setPixel(X, yy, col);
	}
};


// Fonction interne pour placer les 8 pixels symétriques du cercle
void plotCirclePoints(int cx, int cy, int px, int py, uint col) {
	screen.setPixel(cx + px, cy + py, col);
	screen.setPixel(cx - px, cy + py, col);
	screen.setPixel(cx + px, cy - py, col);
	screen.setPixel(cx - px, cy - py, col);
	screen.setPixel(cx + py, cy + px, col);
	screen.setPixel(cx - py, cy + px, col);
	screen.setPixel(cx + py, cy - px, col);
	screen.setPixel(cx - py, cy - px, col);
};


void drawCircle(int xc, int yc, int r, uint color)
{
    if (r <= 0) return;

    int x = 0;
    int y = r;
    int d = 1 - r; // Décision initiale

    // Tracer les points initiaux
    plotCirclePoints(xc, yc, x, y, color);

    // Boucle jusqu’à ce que x > y
    while (x < y)
    {
        x++;
        if (d < 0) {
            // Le pixel de décision était intérieur au cercle
            // On se déplace seulement en x
            d += 2 * x + 1;
        } else {
            // Le pixel de décision était extérieur ou sur le cercle
            // On se déplace en x et en y
            y--;
            d += 2 * (x - y) + 1;
        }

        plotCirclePoints(xc, yc, x, y, color);
    }
}


void fillCircle(int xc, int yc, int r, uint color)
{
    if (r <= 0) return;

    int x = 0;
    int y = r;
    int d = 1 - r;

    // On commence par tracer la ligne au sommet du cercle
    // (yc + y) et (yc - y) peuvent être les mêmes si y=0, mais dans ce cas-ci y=r>0.
    drawHorizontalLine(yc + y, xc - x, xc + x, color); // y haut
    drawHorizontalLine(yc - y, xc - x, xc + x, color); // y bas

    // De même, si x != y, on dessine les lignes horizontales correspondantes
    // à l'échange x<->y (tant que x < y)
    if (x != y) {
        drawHorizontalLine(yc + x, xc - y, xc + y, color);
        drawHorizontalLine(yc - x, xc - y, xc + y, color);
    }

    while (x < y) {
        x++;

        if (d < 0) {
            d += 2 * x + 3;
        } else {
            y--;
            d += 2 * (x - y) + 5;
        }

        // On dessine maintenant les lignes horizontales correspondant
        // aux 8 octants : on connait (x, y) donc on a :
        // (xc ± x, yc ± y) et (xc ± y, yc ± x)
        // On remplit les lignes entre -x et x pour les y correspondants,
        // et entre -y et y pour les x correspondants, ce qui remplit tout le disque.
        
        drawHorizontalLine(yc + y, xc - x, xc + x, color);
        drawHorizontalLine(yc - y, xc - x, xc + x, color);

        if (x != y) {
            drawHorizontalLine(yc + x, xc - y, xc + y, color);
            drawHorizontalLine(yc - x, xc - y, xc + y, color);
        }
    }
}
